<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Turn;
use App\Models\Takings;

class TurnController extends Controller
{
    public function index()
    {
        $turns = Turn::with(['profile', 'workstation'])
                    // ->whereNotNull('end_at')
                    ->orderBy('begin_at', 'desc')
                    ->paginate();
        return response()->json($turns);
    }

    public function activeTurns()
    {
        $turns = Turn::with(['profile', 'workstation'])
                ->withSum('takings', 'amount')
                ->whereNull('end_at')
                ->whereHas('takings.payments', function ($query) {
                    $query->where('payment_method_id', '=', '1');
                })
                ->orderBy('begin_at', 'desc')
                ->paginate();

        
        return response()->json($turns);
    }

    public function searchTurn(Request $request)
    {
        $number = $request->number;
        $turn = Turn::with(['profile', 'workstation'])->where('number', $number)->first();
        if (!$turn){
            return response()->json([
                "status" => 0,
                "message" => "Numero de turno no se encuentra registrado"
            ]);
        }

        return response()->json([
            "status" => 1,
            "message" => "Se esta mostrando la informacion del turno " . $number,
            "item" => $turn
        ]);
    }

    public function show(Request $request)
    {
        $number = $request->number;
        $data = $this->getTurnSumary($number);
        return response()->json($data);
    }

    public function closeTurn(Request $request)
    {
        $number = $request->numberTurn;
        $data = $this->getTurnSumary($number, true);
        return response()->json($data);
    }

    public function lastTakings(Request $request)
    {
        $number = $request->number;
        $turn = Turn::where('number', $number)->first();

        if (!$turn){
            return response()->json([
                "status" => 0,
                "message" => "Numero de turno no se encuentra registrado"
            ]);
        }

        $takings = Takings::with(['payments.paymentMethod', 'vehicleType', 'vehicle'])
                            ->select(['takings.*'])
                            ->where('turn_id', $turn->id)
                            ->take(50)
                            ->orderBy('id', 'desc')->get();

        return response()->json([
            "status" => 1,
            "data" => $takings
        ]);
    }

    private function getTurnSumary($number, $closeTurn = false)
    {
        // Object data will be the response of the function
        $data = new \stdClass();

        // recover the turn data info of the database
        $turn = Turn::with(['takings.payments', 'workstation', 'profile'])->where('number', $number)->first();

        if (!$turn) {
            return response()->json([
                "status" => 0,
                "message" => "Turno no encontrado"
            ]);
        }

        if ($turn->end_at == null && $closeTurn){
            // Close turn
            $now = date('Y-m-d H:i:s');
            $turn->end_at = $now;
            $turn->save();

            // Revoke Tokens
            $turn->profile->user->tokens()->delete();
        }

        // Start recover the info
        // BEGIN Static info
        $data->collector = $turn->profile->names .' '. $turn->profile->surnames;
        $data->workstation = $turn->workstation->name;
        $data->beginAt = $turn->begin_at;
        $data->endAt = $turn->end_at;
        $data->turnNumber = $turn->number;
        // END Static info

        // BEGIN Computed info
        // Vehicle types
        $data->vehicleType = [];
        $data->vehicleTypeAmount = [];
        $data->paymentMethod = [];
        $data->mixPaymentMethod = [];

        for ($i=1; $i < 10; $i++) {
            $data->vehicleType[$i] = 0;
            $data->vehicleTypeAmount[$i] = 0;
            $data->paymentMethod[$i]= 0;
            $data->mixPaymentMethod[$i] = 0;
        }

        foreach ($turn->takings as $taking) {
            $data->vehicleType[$taking->vehicle_type_id]++;
            $data->vehicleTypeAmount[$taking->vehicle_type_id] += floatval($taking->amount);

            foreach ($taking->payments as $payment) {
                $data->paymentMethod[$payment->payment_method_id] += floatval($payment->amount);
            }

            if ($taking->payments->count() > 1) {
                foreach ($taking->payments as $payment) {
                    $data->mixPaymentMethod[$payment->payment_method_id] += floatval($payment->amount);
                }
            }

        }
        return $data;

        // END Computed info
    }

    public function delete(Request $request)
    {
        $id = $request->id;
        try{
            $turn = Turn::find($id);
            $turn->delete();
            return response()->json([
                "status" => 1,
                "message" => "Turno numero ". $turn->number . " eliminado exitosamente"
            ]);

        } catch (\Exception $e) {
            return response()->json([
                "status" => 0,
                "message" => "Ocurrio un error, intente mas tarde, si el problema persiste comuniquese con su administrador"
            ]);
        }
    }
}
